/**
 * \file: AudioSinkChannel.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
 * \component: Android Auto
 *
 * \author: I. Hayashi / ADITJ/SW / ihayashi@jp.adit-jv.com
 *
 * \copyright (c) 2014 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

#include <aauto/AditAudioSink.h>
#include "AudioSinkChannel.h"
#include "DemoFactory.h"

LOG_IMPORT_CONTEXT(demo)

namespace adit { namespace aauto {

AudioSinkChannel::AudioSinkChannel(::shared_ptr<GalReceiver> inReceiver, uint8_t inSessionId,
                                   AudioStreamType inStreamType, IDynamicConfiguration& inConfig) :
        receiver(inReceiver), sessionId(inSessionId), mStreamType(inStreamType), mConfig(&inConfig)
{
    endpoint = nullptr;
}

AudioSinkChannel::~AudioSinkChannel()
{
    endpoint = nullptr;
}

void AudioSinkChannel::shutdown()
{
    LOG_INFO((demo, "audio sink channel shut down"));
    if(endpoint != nullptr) {
        endpoint->shutdown();
    }
}

bool AudioSinkChannel::Initialize()
{
    LOG_INFO((demo, "Creating new Audio sink... (mStreamType=%d)", mStreamType));
    auto creator = DemoFactory::instance().getCreator
            <AditAudioSink, AudioSinkCreatorFn>(mConfig->GetItem("adit-audio-sink", ""));
    if (creator != nullptr)
    {
        endpoint = creator(sessionId, receiver->messageRouter());
    }
    else
    {
        LOG_ERROR((demo, "failed to get creator object"));
    }

    if(endpoint != nullptr)
    {
        LOGD_DEBUG((demo, "Completed the initialization of audio-sink (mStreamType=%d)", mStreamType));
    }
    else
    {
        LOG_ERROR((demo, "error create AudioSinkImpl"));
        return false;
    }

    /* list for configuration values */
    std::list<std::string> audio_sink;

    /* GalReceiver/AIL Configuration */
    if (mStreamType == AUDIO_STREAM_MEDIA)
    {
        /* get list of AudioSink Media configuration values */
        audio_sink = mConfig->GetItems("audio-sink-media");
    }
    else if (mStreamType == AUDIO_STREAM_SYSTEM_AUDIO)
    {
        /* get list of AudioSink System-Audio configuration values */
        audio_sink = mConfig->GetItems("audio-sink-system-audio");
    }
    else if (mStreamType == AUDIO_STREAM_GUIDANCE)
    {
        /* get list of AudioSink Guidance configuration values */
        audio_sink = mConfig->GetItems("audio-sink-guidance");
    }
    else
    {
        LOG_ERROR((demo, "Not supported stream type %d", mStreamType));
        return false;
    }

    for (auto node : audio_sink)
    {
        std::string key;
        std::string value;

        /* If audio-sink-max-unacked-frames is not configured, default value of 4096 is used.
         * In this case, the MD send many audio data packages which might not all handled while playback is active. */

        /* get key and respective value from node entry */
        mConfig->GetNodeItem(node, key, value);

        /* Configuration for AudioSink Endpoints */
        endpoint->setConfigItem(key, value);
    }

    endpoint->registerCallbacks(this);

    if(!endpoint->init())
    {
        LOG_ERROR((demo, "error endpoint->init()"));
        return false;
    }

    LOGD_DEBUG((demo, "mGalReceiver->registerService<AudioSink>"));
    // The argument of API is done const_cast because GalReceiver requires non const pointer.
    return receiver->registerService(const_cast<AditAudioSink*>(endpoint.get()));
}

void AudioSinkChannel::playbackStartCallback(int32_t sessionId)
{
    LOGD_DEBUG((demo, "AudioSinkChannel::playbackStartCallback"));
}

void AudioSinkChannel::playbackStopCallback(int32_t sessionId)
{
    LOGD_DEBUG((demo, "AudioSinkChannel::playbackStopCallback"));
}

void AudioSinkChannel::notifyErrorCallback(aautoErrorCodes inErrorCode)
{
    LOG_ERROR((demo, "AudioSinkChannel::notifyErrorCallback: %d", inErrorCode));
}

} } // namespace adit { namespace aauto
